/** @file   GraphicsContainer.h
 * @brief   Declaration of GraphicsContainer class.
 * @version $Revision: 1.1 $
 * @date    $Date: 2006/02/21 20:45:13 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_GraphicsContainer_H
#define H_ENG2D_GraphicsContainer_H

#include <vector>
#include "BaseContainer.h"
#include "datatypes.h"
#include "eng2d_Exception.h"
#include "eng2d_dll.h"


// Forward declarations.
struct BITMAP;
struct RLE_SPRITE;
struct DATAFILE;

namespace eng2d {

/** @class  GraphicsContainer
 * @brief   A graphic container where you can place several graphic objects.
 * @author  Tomi Lamminsaari
 *
 * A GraphicsContainer container can store either Allegro BITMAP objects or
 * RLE_SPRITE objects. The container can be added to a DataWarehouse
 * where you can find it by its UID.
 */
class DLLIMPORT GraphicsContainer : public BaseContainer
{
public:

    ///
    /// Constants, datatypes and static methods
    /// =======================================


    ///
    /// Constructors, destructor and operators
    /// ======================================

    /** Default constructor.
     * @param   aContentType    Type of the data this item stores.
     */
    GraphicsContainer( GfxContentType aContentType );

    /** Destructor
     */
    virtual ~GraphicsContainer();

    /** Copy constructor
     * @param   aAnother          Reference to another GraphicsContainer
     */
    GraphicsContainer( const GraphicsContainer& aAnother );

    /** An assignment operator
     * @param   aAnother          Reference to another GraphicsContainer
     * @return  Reference to this object.
     */
    GraphicsContainer& operator= ( const GraphicsContainer& aAnother );


    ///
    /// Inherited methods
    /// =================

    /** From WarehouseItem.
     */
    virtual void ResetAndDestroy();
    
    /** From WarehouseItem.
     */
    virtual void Push( void* aObject );
    
    /** From WarehouseItem.
     */
    virtual void* At( int aUid ) const;


    ///
    /// New public methods
    /// ==================
    

    ///
    /// Getter methods
    /// ==============

    /** Returns the index'th bitmap stored by this WarehouseItem.
     * @param   aIndex          Index of the bitmap in this item.
     * @return  Pointer to allegro bitmap. Ownerhip is not transferred.
     * @exception xOutOfBounds  Thrown if index was invalid.
     * @exception xLogicError   Thrown if content type is wrong.
     */
    virtual BITMAP* GetBitmap( int aIndex ) const throw ( xOutOfBounds,
                                                          xLogicError );
    
    /** Returns the index'th rle sprite stored by this WarehouseItem.
     * @param   aIndex          Index of the rle sprite.
     * @return  Pointer to allegro RLE_SPRITE. A null pointer will be returned
     *          if contents are not rle sprites. Ownership is not transferred.
     * @exception xOutOfBounds  Thrown if index was invalid.
     * @exception xLogixError   Thrown if content type is wrong.
     */
    virtual RLE_SPRITE* GetRleSprite( int aIndex ) const throw ( xOutOfBounds,
                                                                 xLogicError );
    
    /** Returns the type of the graphic contents stored by this WarehouseItem.
     * @return  Type of the graphic contents.
     */
    GfxContentType ContentType() const;
    
    /** Returns the number of bitmaps/rle-sprites this item holds.
     * @return  Number of graphic objects this item contains.
     */
    int GraphicCount() const;
    
protected:
    ///
    /// Protected interface
    /// ===================


protected:
    ///
    /// Data members
    /// ============

    /** Type of the contents stored by this item. */
    GfxContentType      iContentType;
    
    /** An array of pointers to bitmaps/rle-sprites. */
    std::vector<void*>  iGraphics;
};

};
#endif
